/*
 * @(#)AMControlFlow.java  1.0  2006-12-17
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.am.model;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import java.util.*;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;

/**
 * Represents the MODEL (structural, behavioral representation) of an
 * ControlFlow element in an Activity Diagram.
 * <p>
 * A control flow is an edge that starts an activity node after the previous
 * one is finished. Objects and data cannot pass along a control flow edge.
 * <p>
 * A control flow is notated by an arrowed line connecting two nodes.
 * <p>
 * For further information, consult the official UML
 * <a href="http://www.omg.org/docs/formal/05-07-04.pdf">online resource</a>
 * of the Object Management Group (OMG):<br>
 * Unified Modeling Language: Superstructure, version 2.0, formal/05-07-04,
 * page 344 et sqq.
 *
 * @author Florian Padrun
 * @version 1.0 2006-12-17 Created.
 */
public class AMControlFlow extends AbstractRelationship {
    /** Name of the guardCondition. */
    private String guardCondition
            = ActivityModel.labels.getString("guardConditionDefaultName");

    /** Specifies whether isGuarded is enabled; the default is false. */
    private boolean isGuarded = false;
    
    /** Creates a new instance. */
    public AMControlFlow() {
    }
    
    /**
     * Sets isGuarded.
     * Used to show/hide the guardCondition of a control flow.
     * @param newValue boolean
     */
    public void setGuarded(boolean newValue) {
        boolean oldValue = isGuarded;
        isGuarded = newValue;
        firePropertyChange("isGuarded", oldValue, newValue);
    }
    
    /**
     * Gets isGuarded.
     * Used to show/hide the guardCondition of a control flow.
     * @return isGuarded
     */
    public boolean isGuarded() {
        return isGuarded;
    }
    
    /**
     * Sets the name of the guardCondition.
     * @param newValue String
     */
    public void setGuardCondition(String newValue) {
        String oldValue = guardCondition;
        guardCondition = newValue;
        firePropertyChange("guardCondition", oldValue, newValue);
    }
    
    /**
     * Gets the name of the guardCondition.
     * @return name
     */
    public String getGuardCondition() {
        return guardCondition;
    }
    
    /**
     * Returns an ID specifying the concept simulated by this element.
     * @return CONTROL_FLOW
     */
    @Override
    public int getSimulatedConcept() {
        return ActivityModel.CONTROL_FLOW;
    }
    
    /**
     * Returns true if this relationship can connect the provided start and end
     * elements.
     * @param start SimulatedElement
     * @param end SimulatedElement
     */
    @Override
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        int startElement = start.getSimulatedConcept();
        int endElement = end.getSimulatedConcept();
        
        /*
         * An InitialNode can never be an endElement.
         * An ActivityFinalNode can never be a startElement.
         */
        if ((startElement == ActivityModel.INITIAL_NODE ||
                startElement == ActivityModel.ACTION ||
                startElement == ActivityModel.DECISION_NODE)
                &&
                (endElement == ActivityModel.ACTION ||
                endElement == ActivityModel.DECISION_NODE ||
                endElement == ActivityModel.ACTIVITY_FINAL_NODE)) {
            return true;
        }
        return false;
    }
    
    /**
     * Returns true if this relationship can connect the provided start element.
     * @param start SimulatedElement
     */
    @Override
    public boolean canConnect(SimulatedElement start) {
        switch (start.getSimulatedConcept()) {
            case ActivityModel.ACTION :
            case ActivityModel.ACTIVITY_FINAL_NODE :
            case ActivityModel.DECISION_NODE :
            case ActivityModel.INITIAL_NODE :
                return true;
            default :
                return false;
        }
    }
    
    /**
     * Creates and returns a copy of the model object.
     * @return that
     */
    @Override
    public AMControlFlow clone() {
        AMControlFlow that = (AMControlFlow) super.clone();
        return that;
    }
    
    /**
     * Reads the persistent model object.
     * @param in DOMInput
     */
    @Override
    public void read(DOMInput in) throws IOException {
        ResourceBundleUtil labels
                = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault());
        name = in.getAttribute("name",
                ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault()).getString("controlFlowDefaultName"));
        guardCondition = in.getAttribute("guardCondition",
                ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault()).getString("guardConditionDefaultName"));
        
        // Read the start of a ControlFlow connection
        in.openElement("Start");
        if (in.getElementCount("AMInitialNode") == 1) {
            in.openElement("AMInitialNode");
            setStart((AMInitialNode) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMAction") == 1) {
            in.openElement("AMAction");
            setStart((AMAction) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMDecisionNode") == 1) {
            in.openElement("AMDecisionNode");
            setStart((AMDecisionNode) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMActivityFinalNode") == 1) {
            in.openElement("AMActivityFinalNode");
            setStart((AMActivityFinalNode) in.readObject());
            in.closeElement();
        }
        in.closeElement();
        
        // Read the end of a ControlFlow connection
        in.openElement("End");
        if (in.getElementCount("AMInitialNode") == 1) {
            in.openElement("AMInitialNode");
            setEnd((AMInitialNode) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMAction") == 1) {
            in.openElement("AMAction");
            setEnd((AMAction) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMDecisionNode") == 1) {
            in.openElement("AMDecisionNode");
            setEnd((AMDecisionNode) in.readObject());
            in.closeElement();
        } else if (in.getElementCount("AMActivityFinalNode") == 1) {
            in.openElement("AMActivityFinalNode");
            setEnd((AMActivityFinalNode) in.readObject());
            in.closeElement();
        }
        in.closeElement();
    }
    
    /**
     * Writes the model object in a persisten state.
     * @param out DOMOutput
     */
    @Override
    public void write(DOMOutput out) throws IOException {
        out.addAttribute("name", name,
                ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault()).getString("controlFlowDefaultName"));
        out.addAttribute("guardCondition", guardCondition,
                ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault()).getString("guardConditionDefaultName"));
        
        // Write the start of a ControlFlow connection
        out.openElement("Start");
        out.writeObject(getStart());
        out.closeElement();
        
        // Write the end of a ControlFlow connection
        out.openElement("End");
        out.writeObject(getEnd());
        out.closeElement();
    }
    
    /**
     * Handles the disconnection of a connection.
     * @param start SimulatedElement
     * @param end SimulatedElement
     */
    @Override
    protected void handleDisconnecting(SimulatedElement start,
            SimulatedElement end) {
        int startElement =  start.getSimulatedConcept();
        int endElement = end.getSimulatedConcept();
        
        switch (startElement) {
            case (ActivityModel.INITIAL_NODE):
            {
                AMInitialNode s = (AMInitialNode) start;
                break;
            }
            case (ActivityModel.ACTION):
            {
                AMAction s = (AMAction) start;
                break;
            }
            case (ActivityModel.DECISION_NODE):
            {
                AMDecisionNode s = (AMDecisionNode) start;
                break;
            }
        }
        
        switch (endElement) {
            case (ActivityModel.ACTION):
            {
                AMAction e = (AMAction) end;
                break;
            }
            case (ActivityModel.DECISION_NODE):
            {
                AMDecisionNode e = (AMDecisionNode) end;
                break;
            }
            case (ActivityModel.ACTIVITY_FINAL_NODE):
            {
                AMActivityFinalNode e = (AMActivityFinalNode) end;
                break;
            }
        }
    }
    
    /**
     * Handles the connection of a connection.
     * @param start SimulatedElement
     * @param end SimulatedElement
     */
    @Override
    protected void handleConnecting(SimulatedElement start,
            SimulatedElement end) {
        int startElement = start.getSimulatedConcept();
        int endElement = end.getSimulatedConcept();
        
        switch (startElement) {
            case (ActivityModel.INITIAL_NODE):
            {
                AMInitialNode s = (AMInitialNode) start;
                break;
            }
            case (ActivityModel.ACTION):
            {
                AMAction s = (AMAction) start;
                break;
            }
            case (ActivityModel.DECISION_NODE):
            {
                AMDecisionNode s = (AMDecisionNode) start;
                break;
            }
        }
        
        switch (endElement) {
            case (ActivityModel.ACTION):
            {
                AMAction e = (AMAction) end;
                break;
            }
            case (ActivityModel.DECISION_NODE):
            {
                AMDecisionNode e = (AMDecisionNode) end;
                break;
            }
            case (ActivityModel.ACTIVITY_FINAL_NODE):
            {
                AMActivityFinalNode e = (AMActivityFinalNode) end;
                break;
            }
        }
    }
}